/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.internal.tasks.testing.failure.mappers;

import org.gradle.api.internal.tasks.testing.failure.TestFailureMapper;
import org.gradle.api.internal.tasks.testing.failure.ThrowableToTestFailureMapper;
import org.gradle.api.tasks.testing.TestFailure;

import javax.annotation.Nullable;
import java.util.Collections;
import java.util.List;


/**
 * Maps {@code org.opentest4j.AssertionFailedError} to {@link TestFailure}.
 * <p>
 * This mapper is a bit more complex, as depending on the type of the expected and actual values,
 * the assertion will be mapped as a string-based comparison failure or a file comparison failure.
 * <p>
 * See {@link TestFailureMapper} for more details about failure mapping.
 */
public class OpenTestAssertionFailedMapper extends TestFailureMapper {

    @Override
    protected List<String> getSupportedClassNames() {
        return Collections.singletonList("org.opentest4j.AssertionFailedError");
    }

    public TestFailure map(Throwable throwable, ThrowableToTestFailureMapper rootMapper) throws Exception {
        Object expectedValueWrapper = invokeMethod(throwable, "getExpected", Object.class);
        Object expectedValue = invokeMethod(expectedValueWrapper, "getValue", Object.class);

        Object actualValueWrapper = invokeMethod(throwable, "getActual", Object.class);
        Object actualValue = invokeMethod(actualValueWrapper, "getValue", Object.class);

        if (expectedValue.getClass().getName().equals("org.opentest4j.FileInfo") || actualValue.getClass().getName().equals("org.opentest4j.FileInfo")) {
            return mapFileInfoComparisonFailure(throwable, expectedValue, actualValue);
        } else {
            return mapStringBasedComparisonFailure(throwable, expectedValueWrapper, actualValueWrapper);
        }
    }

    private static TestFailure mapFileInfoComparisonFailure(Throwable throwable, Object expected, Object actual) throws Exception {
        String expectedPath = extractValue(expected);
        byte[] expectedContent = extractContent(expected);
        String actualPath = extractValue(actual);
        byte[] actualContent = extractContent(actual);
        return TestFailure.fromFileComparisonFailure(throwable, expectedPath, actualPath, expectedContent, actualContent, null);
    }

    private static TestFailure mapStringBasedComparisonFailure(Throwable throwable, Object expected, Object actual) throws Exception {
        String expectedValue = extractValue(expected);
        String actualValue = extractValue(actual);
        return TestFailure.fromTestAssertionFailure(throwable, expectedValue, actualValue, null);
    }

    @Nullable
    private static String extractValue(@Nullable Object valueWrapper) throws Exception {
        if (valueWrapper == null) {
            return null;
        }

        if (valueWrapper.getClass().getName().equals("org.opentest4j.FileInfo")) {
            return invokeMethod(valueWrapper, "getPath", String.class);
        } else if (valueWrapper.getClass().getName().equals("org.opentest4j.ValueWrapper")) {
            return invokeMethod(valueWrapper, "getStringRepresentation", String.class);
        } else if (valueWrapper.getClass().getName().equals("java.lang.String")) {
            return (String) valueWrapper;
        } else {
            return null;
        }
    }

    @Nullable
    private static byte[] extractContent(@Nullable Object valueWrapper) throws Exception {
        if (valueWrapper == null) {
            return null;
        }

        if (valueWrapper.getClass().getName().equals("org.opentest4j.FileInfo")) {
            return invokeMethod(valueWrapper, "getContents", byte[].class);
        } else {
            return null;
        }
    }

}
