/////////////////////////////////////////////////////////////////////////////

#include "headers.h"

#include "rip.h"
#include "decode.h"
#include "wav.h"

/////////////////////////////////////////////////////////////////////////////
//
// returns the number of errors
//
int rip(
  FILE *inf,
  struct OPTIONS *opts,
  rip_blockdetect_cb_t blockdetect_cb
) {
  struct OPTIONS myopts;
  char *myoutfilename = NULL;
  int rv = 1;
  uint8 *block = NULL;
  uint8 *block_r = NULL;
  sint16 *outbuf = NULL;
  sint16 *outbuf_r = NULL;
  FILE *outf = NULL;
  int p1[2] = {0, 0};
  int p2[2] = {0, 0};
  long in_pos, out_indicator_pos;
  uint32 bytes_consumed = 0;
  uint32 readunit;
  int endflag = 0;

  in_pos = 0;
  if(inf) in_pos = ftell(inf);

  // copy options locally, or make a blank version if it wasn't given
  if(opts) { memcpy(&myopts, opts, sizeof(myopts)); }
  else { memset(&myopts, 0, sizeof(myopts)); }

  // use a default for the output filename if it doesn't exist
  if(!myopts.outname) { myopts.outname = "out"; }

  { int outfilename_l = strlen(myopts.outname);
    myoutfilename = malloc(outfilename_l + 4 + 1);
    if(!myoutfilename) {
      printf("unable to allocate string buffer");
      goto error;
    }
    strcpy(myoutfilename, myopts.outname);
    if(
      (outfilename_l < 4) ||
      (strcasecmp(myoutfilename + outfilename_l - 4, ".wav"))
    ) {
      strcat(myoutfilename, ".wav");
    }
  }
  printf("writing %s: ", myoutfilename); fflush(stdout);

  // make sure an input file was given
  if(!inf) {
    printf("no input file opened");
    goto error;
  }

  // fill in and validate all parts of myopts

  // channel count
  if(!myopts.use_channels) myopts.channels = 1;
  if(myopts.channels < 1 || myopts.channels > 2) {
    printf("invalid channel count (%d)", myopts.channels);
    goto error;
  }

  // honor end flag
  if(!myopts.use_honor_endflag) myopts.honor_endflag = 1;

  // max bytes will be used/checked later

  // sample rate
  if(!myopts.use_samplerate) myopts.samplerate = 44100;

  // offset
  if(!myopts.use_offset) myopts.offset = 0;

  // interleave size
  // defaults to 16K for stereo, 16 bytes (the minimum) for mono
  if(!myopts.use_interleave) {
    myopts.interleave = (myopts.channels == 2) ? 0x4000 : 0x10;
  }
  if((myopts.interleave < 1) || ((myopts.interleave) & 0xF)) {
    printf(
      "invalid interleave size (%d), must be a positive multiple of 16",
      myopts.interleave
    );
    goto error;
  }

  // skip
  if(!myopts.use_skip) myopts.skip = 0;

  //
  // now apply the starting offset if necessary
  //
  if(myopts.offset) fseek(inf, myopts.offset, SEEK_CUR);

  //
  // allocate input/output blocks
  //
  readunit = ((uint32)(myopts.interleave)) * ((uint32)(myopts.channels));

  block = malloc(readunit);
  if(!block) { printf("unable to allocate block buffer"); goto error; }
  outbuf = malloc((readunit / 0x10) * 28 * sizeof(sint16));
  if(!outbuf) { printf("unable to allocate output buffer"); goto error; }

  block_r = block;
  outbuf_r = outbuf;
  if(myopts.channels == 2) {
    block_r = block + myopts.interleave;
    outbuf_r = outbuf + (myopts.interleave / 0x10) * 28;
  }

  //
  // now, open the output wav file
  //
  outf = wav_create(myoutfilename, myopts.channels, myopts.samplerate);
  if(!outf) { printf("%s", strerror(errno)); goto error; }

  out_indicator_pos = ftell(outf);

  bytes_consumed = 0;
  endflag = 0;
  for(;;) {
    int isfirst = 0;
    uint8 outline[28 * 2 * 2];
    uint32 j;
    uint32 l, lines;
    int r;

    isfirst = (!bytes_consumed);

    in_pos = ftell(inf);

    if(endflag) break;

    if(myopts.use_maxbytes) {
      if(bytes_consumed >= myopts.maxbytes) break;
      if((bytes_consumed + readunit) > myopts.maxbytes) break;
    }

    r = fread(block, 1, readunit, inf);
    if(r < 0) {
      printf("%s", strerror(errno));
      break;
    }
    if(!r) break;
    if(r < readunit) break;

    bytes_consumed += readunit;

    lines = myopts.interleave / 0x10;
    if((!isfirst) && (blockdetect_cb)) {
      if(blockdetect_cb(in_pos, block, readunit)) break;
    }

    if(myopts.honor_endflag) {
      for(l = 0; l < lines; l++) {
        if(
          (block  [0x10 * l + 1] & 1) ||
          (block_r[0x10 * l + 1] & 1)
        ) {
          endflag = 1;
          lines = l + 1;
          break;
        }
      }
    }

    if(lines < 1) break;

    if(myopts.channels == 1) {
      decode(block  , outbuf  , p1  , p2  , lines);
      for(l = 0; l < lines; l++) {
        for(j = 0; j < 28; j++) {
          outline[2 * j + 0] = outbuf[28 * l + j];
          outline[2 * j + 1] = outbuf[28 * l + j] >> 8;
        }
        fwrite(outline, 1, 28 * 2, outf);
      }
    } else {
      decode(block  , outbuf  , p1+0, p2+0, lines);
      decode(block_r, outbuf_r, p1+1, p2+1, lines);
      for(l = 0; l < lines; l++) {
        for(j = 0; j < 28; j++) {
          outline[4 * j + 0] = outbuf  [28 * l + j];
          outline[4 * j + 1] = outbuf  [28 * l + j] >> 8;
          outline[4 * j + 2] = outbuf_r[28 * l + j];
          outline[4 * j + 3] = outbuf_r[28 * l + j] >> 8;
        }
        fwrite(outline, 1, 28 * 2 * 2, outf);
      }
    }

    // progress indicator
    if((out_indicator_pos >> 20) != (ftell(outf) >> 20)) {
      out_indicator_pos = ftell(outf);
      printf("."); fflush(stdout);
    }

    // skip blocks if several channels are being interleaved
    if(myopts.skip) fseek(inf, myopts.skip, SEEK_CUR);

  }

  printf("ok");

  goto success;

error:   rv = 1; goto end;
success: rv = 0; goto end;
end:
  if(outf) wav_close(outf);
  if(myoutfilename) free(myoutfilename);
  if(block ) free(block );
  if(outbuf) free(outbuf);

  if(inf) fseek(inf, in_pos, SEEK_SET);

  printf("\n");
  fflush(stdout);
  return rv;
}

/////////////////////////////////////////////////////////////////////////////
