/////////////////////////////////////////////////////////////////////////////

#include "headers.h"

#include "fmt.h"

/////////////////////////////////////////////////////////////////////////////

typedef int (*fmt_rip_t)(FILE *inf, struct OPTIONS *opts);
typedef int (*fmt_detect_t)(FILE *inf, const char *ext);

/////////////////////////////////////////////////////////////////////////////

struct FMT_TABLE {
  const char *name;
  fmt_rip_t rip;
  fmt_detect_t detect;
  const char *desc;
};

/////////////////////////////////////////////////////////////////////////////

static const struct FMT_TABLE fmt_table[] = {
  { "raw", fmt_raw_rip, fmt_raw_detect, "Raw data (for experimentation)" },
  { "vb2", fmt_vb2_rip, fmt_vb2_detect, "Konami multi-song .BIN/.VB2 file (Dance Dance Revolution, etc.)" },
  { "8"  , fmt_8_rip  , fmt_8_detect  , ".8 file (Suikoden)" },
  { "msa", fmt_msa_rip, fmt_msa_detect, ".MSA file (Psyvariar, possibly other PS2 games)" },
  { "xa2", fmt_xa2_rip, fmt_xa2_detect, ".XA2 file (Extreme-G 3, possibly other PS2 games)" },
  { NULL, NULL, NULL }
};

/////////////////////////////////////////////////////////////////////////////

void fmt_printtable(void) {
  const struct FMT_TABLE *t;
  printf("  Format   Description\n");
  printf("  ------------------------------------------------------------\n");
  for(t = fmt_table; t->name; t++) {
    printf("  %-9s%s\n", t->name, t->desc);
  }
}

/////////////////////////////////////////////////////////////////////////////
//
// determine if a given format name is recognized
//
int fmt_isvalid(const char *fmt) {
  const struct FMT_TABLE *t;
  for(t = fmt_table; t->name; t++) {
    if(!strcmp(t->name, fmt)) return 1;
  }
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//
// detect the format of a file if none was given
// (must return at least some valid string pointer)
//
const char *fmt_detect(FILE *inf, const char *ext) {
  const struct FMT_TABLE *t;
  const char *fmt_highest = "raw";
  int fmt_highest_score = 50;
  for(t = fmt_table; t->name; t++) {
    if(t->detect) {
      int score = t->detect(inf, ext);
      if(score > fmt_highest_score) {
        fmt_highest = t->name;
        fmt_highest_score = score;
      }
    }
  }
  return fmt_highest;
}

/////////////////////////////////////////////////////////////////////////////
//
// rip the actual music from a file
// returns the number of errors that occurred
// fmt and outname must be valid in opts
//
int fmt_rip(FILE *inf, struct OPTIONS *opts) {
  const struct FMT_TABLE *t;

  struct OPTIONS myopts;

  if(opts) { memcpy(&myopts, opts, sizeof(myopts)); }
  else { memset(&myopts, 0, sizeof(myopts)); }

  if(!inf) {
    printf("no input file given\n");
    return 1;
  }

  if(!myopts.fmt) { myopts.fmt = "raw"; }
  if(!myopts.outname) { myopts.outname = "out"; }

  if(myopts.use_offset) {
    fseek(inf, myopts.offset, SEEK_CUR);
    myopts.use_offset = 0;
    myopts.offset = 0;
  }

  for(t = fmt_table; t->name; t++) {
    if(!strcmp(t->name, myopts.fmt)) {
      if(t->rip) {
        return t->rip(inf, &myopts);
      } else {
        printf("function missing for format '%s'\n", myopts.fmt);
        return 1;
      }
    }
  }

  printf("unknown format '%s'\n", myopts.fmt);
  return 1;
}

/////////////////////////////////////////////////////////////////////////////
